-- Copyright 2024 by Todd Hundersmarck (ThundR)
-- All Rights Reserved

local thModName = g_currentModName
local thModPath = g_currentModDirectory
THSpecManager = {}
local THSpecManager_mt = Class(THSpecManager)
_G.THSpecType = {
    VEHICLE = 1,
    PLACEABLE = 2
}
function THSpecManager.new(customMt)
    customMt = customMt or THSpecManager_mt
    if THUtils.argIsValid(type(customMt) == THValueType.TABLE, "customMt", customMt) then
        local self = setmetatable({}, customMt)
        self.modName = thModName
        self.modPath = thModPath
        self.thModManager = g_thModManager
        self.specializations = {}
        self.specTypes = {}
        self.specTypeMapping = {}
        for typeId, typeIndex in pairs(THSpecType) do
            local specTypeData = {
                id = typeId,
                index = typeIndex,
                name = "object",
                namePlural = "objects"
            }
            if typeIndex == THSpecType.VEHICLE then
                specTypeData.name = "vehicle"
                specTypeData.namePlural = "vehicles"
                self.specTypeMapping[specTypeData.name] = typeIndex
            elseif typeIndex == THSpecType.PLACEABLE then
                specTypeData.name = "placeable"
                specTypeData.namePlural = "placeables"
                self.specTypeMapping[specTypeData.name] = typeIndex
            end
            self.specTypes[typeIndex] = specTypeData
            self.specializations[typeIndex] = {}
        end
        return self
    end
end
function THSpecManager.addSpecialization(self, specType, specName, className, modName, filename)
    modName = modName or self.modName
    local modData = self.thModManager:getLoadedMod(modName)
    local specTypeData = nil
    if type(specType) == THValueType.STRING then
        local specTypeIndex = self.specTypeMapping[specType]
        if specTypeIndex ~= nil then
            specTypeData = self.specTypes[specTypeIndex]
        end
    elseif specType ~= nil then
        specTypeData = self.specTypes[specType]
    end
    if THUtils.argIsValid(specTypeData ~= nil, "specType", specType)
        and THUtils.argIsValid(type(specName) == THValueType.STRING and specName ~= "", "specName", specName)
        and THUtils.argIsValid(type(className) == THValueType.STRING, "className", className)
        and THUtils.argIsValid(modData ~= nil, "modName", modName)
        and THUtils.argIsValid(type(filename) == THValueType.STRING, "filename", filename)
    then
        local manager = nil
        if specTypeData.index == THSpecType.VEHICLE then
            manager = g_specializationManager
        elseif specTypeData.index == THSpecType.PLACEABLE then
            manager = g_placeableSpecializationManager
        else
            THUtils.errorMsg(nil, "Could not find %s specialization manager", specTypeData.name)
            return false
        end
        if manager ~= nil then
            local absSpecName = modName .. "." .. specName
            local absClassName = modName .. "." .. className
            local absFilename = Utils.getFilename(filename, self.modPath)
            if absFilename == nil or not fileExists(absFilename) then
                THUtils.errorMsg(nil, THMessage.FILE_NOT_FOUND, absFilename)
                return false
            end
            if manager:getSpecializationByName(specName) then
                THUtils.errorMsg(nil, "Duplicate %s specialization found: %s", specTypeData.name, specName)
                return false
            end
            local specData = {
                name = specName,
                absName = absSpecName,
                className = className,
                absClassName = absClassName,
                filename = filename,
                absFilename = absFilename,
                specType = specTypeData,
                modName = modName
            }
            manager:addSpecialization(specName, className, absFilename, modName)
            if manager:getSpecializationByName(specName) then
                local specClass = _G[className]
                if specClass == nil then
                    THUtils.errorMsg(nil, "Cannot find specialization %q class table: %s", specName, className)
                    return false
                end
                specData.class = specClass
                table.insert(self.specializations[specTypeData.index], specData)
                THUtils.displayMsg("Added %s specialization: %s", specTypeData.name, specName)
                return true
            end
        end
    end
    return false
end
function THSpecManager.inj_finalizeTypes(self, superFunc, manager, ...)
    THUtils.call(function()
        local specTypeData = self.specTypes[self.specTypeMapping[manager.typeName]]
        if specTypeData ~= nil then
            local specsArray = self.specializations[specTypeData.index]
            local typedObjects = manager:getTypes()
            if specsArray ~= nil and typedObjects ~= nil then
                for specIndex = 1, #specsArray do
                    local specData = specsArray[specIndex]
                    local timesAdded = 0
                    for objectTypeName, objectType in pairs(typedObjects) do
                        if objectType.specializations ~= nil
                            and not SpecializationUtil.hasSpecialization(specData.class, objectType.specializations)
                            and specData.class.prerequisitesPresent(objectType.specializations)
                        then
                            manager:addSpecialization(objectTypeName, specData.absName)
                            if SpecializationUtil.hasSpecialization(specData.class, objectType.specializations) then
                                timesAdded = timesAdded + 1
                            end
                        end
                    end
                    if timesAdded > 0 then
                        THUtils.displayMsg("Specialization %q added to %d %s", specData.name, timesAdded, specTypeData.namePlural)
                    end
                end
            end
        end
    end)
    return superFunc(manager, ...)
end
local thSpecManager = THSpecManager.new()
if thSpecManager ~= nil then
    _G.g_thSpecManager = thSpecManager
    THUtils.setFunctionHook("TypeManager", "finalizeTypes", false, false, thSpecManager, THSpecManager.inj_finalizeTypes)
end