-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

local thModName = g_currentModName
local thModPath = g_currentModDirectory
source(thModPath .. "scripts/utilities/THUtils.lua")
source(thModPath .. "scripts/utilities/THDebugUtil.lua")
source(thModPath .. "scripts/managers/THModManager.lua")
source(thModPath .. "scripts/managers/THSpecManager.lua")
source(thModPath .. "scripts/objects/THObject.lua")
source(thModPath .. "scripts/managers/THEventManager.lua")
source(thModPath .. "scripts/misc/THInfoLayer.lua")
THFruitState = {
    ZERO = 1,
    MIN = 2,
    GROWTH = 3,
    FORAGE = 4,
    HARVEST = 5,
    WITHERED = 6,
    PREPARING = 7,
    PREPARED = 8,
    CUT = 9,
    EXTRA = 10,
    MAX = 11
}
THUtils.createEnumTable(THFruitState)
THCore = {}
local THCore_mt = THUtils.createClass(THCore)
function THCore.registerXMLPaths(xmlSchema, xmlPath)
end
function THCore.new(dataKey, modName, customMt)
    modName = modName or thModName
    customMt = customMt or THCore_mt
    local modData = g_thModManager:getLoadedMod(modName)
    local modEnv = nil
    if modData ~= nil then
        modEnv = modData.environment
    end
    if THUtils.argIsValid(THUtils.validateId(dataKey) == true, "dataKey", dataKey)
        and THUtils.argIsValid(modEnv ~= nil, "modName", modName)
        and THUtils.argIsValid(type(customMt) == THValueType.TABLE, "customMt", customMt)
    then
        local self = setmetatable({}, customMt)
        self.isServer = g_server ~= nil
        self.isClient = g_client ~= nil
        self.modData = modData
        self.modName = modData.name
        self.modPath = modData.path
        self.dataKey = dataKey
        self.xmlKey = dataKey
        self.thModManager = g_thModManager
        self.thSpecManager = g_thSpecManager
        self.thEventManager = g_thEventManager
        self.i18n = modEnv.g_i18n
        self.inputManager = modEnv.g_inputBinding
        self.i3DManager = modEnv.g_i3DManager
        self.fillTypeManager = modEnv.g_fillTypeManager
        self.fruitTypeManager = modEnv.g_fruitTypeManager
        self.fillTypes = {}
        self.fruitTypes = {}
        self.groundTypes = {}
        self.sprayTypes = {}
        self.chopperTypes = {}
        self.maxFruitState = 0
        self.maxFruitHaulmState = 0
        self.validFruitStateNames = {}
        self.isPreInitFinished = false
        self.hasMapConfig = false
        local mapSettingsModName = string.format("FS%d_TH_MapSettings", THGameVersion)
        local mapSettingsModData = self.thModManager:getLoadedMod(mapSettingsModName)
        if mapSettingsModData ~= nil then
            self.thMapSettings = {
                modName = mapSettingsModData.name,
                modPath = mapSettingsModData.path,
                modData = mapSettingsModData,
                xmlFilename = THUtils.getFilename("maps/mapSettings.xml", mapSettingsModData.path),
                xmlDataKey = "thMapSettings",
                xmlRootKey = "thMapSettings"
            }
        end
        THDebugUtil.createConsoleCommands(self)
        self.thEventManager:addEventListener(self, self.modName)
        return self
    end
end
function THCore.initFillTypeData(self)
    local fillTypeManager = self.fillTypeManager
    local fillTypesArray = fillTypeManager:getFillTypes()
    THUtils.clearTable(self.fillTypes)
    for fillTypeIndex = 1, #fillTypesArray do
        local fillTypeDesc = fillTypesArray[fillTypeIndex]
        local fillTypeId = string.upper(fillTypeDesc.name)
        local thFillTypeData = {
            id = fillTypeId,
            name = fillTypeDesc.name,
            title = fillTypeDesc.title,
            index = fillTypeIndex,
            desc = fillTypeDesc
        }
        self.fillTypes[fillTypeIndex] = thFillTypeData
    end
    return true
end
function THCore.initFruitTypeData(self)
    local fruitTypeManager = self.fruitTypeManager
    local fruitTypesArray = fruitTypeManager:getFruitTypes()
    self.maxFruitState = 0
    self.maxFruitHaulmState = 0
    THUtils.clearTable(self.fruitTypes)
    for fruitTypeIndex = 1, #fruitTypesArray do
        local fruitTypeDesc = fruitTypesArray[fruitTypeIndex]
        local thFruitTypeData = {
            id = fruitTypeDesc.name:upper(),
            name = fruitTypeDesc.name,
            index = fruitTypeIndex,
            title = fruitTypeDesc.name,
            desc = fruitTypeDesc,
            maxNumStates = 0,
            minGrowthState = 0,
            maxGrowthState = 0,
            minForageState = 0,
            maxForageState = 0,
            minPreparingState = 0,
            maxPreparingState = 0,
            minPreparedState = 0,
            maxPreparedState = 0,
            minHarvestState = 0,
            maxHarvestState = 0,
            minWitheredState = 0,
            maxWitheredState = 0,
            minCutState = 0,
            maxCutState = 0,
            cutStates = {},
            minExtraState = 0,
            maxExtraState = 0
        }
        local fruitTypeYield = THUtils.toNumber(fruitTypeDesc.literPerSqm) or 0
        fruitTypeYield = math.max(0, fruitTypeYield)
        thFruitTypeData.yield = fruitTypeYield
        local fillTypeDesc = fruitTypeDesc.fillType
        if fillTypeDesc == nil then
            THUtils.errorMsg(false, "Cannot find fill type associated with fruit type %q [%s], trying alterative method...", thFruitTypeData.name, thFruitTypeData.index)
            fillTypeDesc = self.fillTypeManager:getFillTypeByName(thFruitTypeData.name)
            if fillTypeDesc == nil then
                THUtils.errorMsg(nil, "No fill type associated with fruit type %q [%s], which could lead to unwanted behavior", thFruitTypeData.name, thFruitTypeData.index)
            end
        end
        if fillTypeDesc ~= nil then
            local thFillTypeData = self:getFillType(fillTypeDesc.index)
            if thFillTypeData ~= nil then
                thFruitTypeData.title = thFillTypeData.title or fillTypeDesc.title or fillTypeDesc.name
                thFruitTypeData.fillType = thFillTypeData
                thFruitTypeData.fillTypeIndex = thFillTypeData.index or fillTypeDesc.index
            else
                THUtils.errorMsg(nil, "Invalid fruit fillType: %s [%s]", fillTypeDesc.name, fillTypeDesc.index)
                thFruitTypeData.title = fillTypeDesc.title or fillTypeDesc.name
                thFruitTypeData.fillTypeIndex = fillTypeDesc.index
            end
        end
        local windrowDesc = fruitTypeDesc.windrowFillType
        if windrowDesc == nil then
            if fruitTypeDesc.hasWindrow and fruitTypeDesc.windrowName ~= nil then
                windrowDesc = self.fillTypeManager:getFillTypeByName(fruitTypeDesc.windrowName)
            end
        end
        if windrowDesc ~= nil then
            local thFillTypeData = self:getFillType(windrowDesc.index)
            if thFillTypeData ~= nil then
                thFruitTypeData.windrowFillType = thFillTypeData
                thFruitTypeData.windrowIndex = thFillTypeData.index
            else
                thFruitTypeData.windrowIndex = windrowDesc.index
            end
            local windrowYield = THUtils.toNumber(fruitTypeDesc.windrowLiterPerSqm) or -1
            if windrowYield >= 0 then
                if fruitTypeYield <= 0 then
                    thFruitTypeData.windrowFactor = 0
                else
                    thFruitTypeData.windrowFactor = windrowYield / fruitTypeYield
                end
                thFruitTypeData.windrowYield = windrowYield
            else
                thFruitTypeData.windrowYield = fruitTypeYield
                thFruitTypeData.windrowFactor = 1
            end
        end
        local windrowCutDesc = fruitTypeDesc.windrowCutFillType
        if windrowCutDesc ~= nil then
            local thFillTypeData = self:getFillType(windrowCutDesc.index)
            if thFillTypeData ~= nil then
                thFruitTypeData.windrowCutFillType = thFillTypeData
                thFruitTypeData.windrowCutIndex = thFillTypeData.index
            else
                thFruitTypeData.windrowCutIndex = windrowCutDesc.index
            end
            local windrowCutFactor = THUtils.toNumber(fruitTypeDesc.windrowCutFactor) or -1
            if windrowCutFactor >= 0 then
                if fruitTypeYield <= 0 then
                    thFruitTypeData.windrowCutYield = 0
                else
                    thFruitTypeData.windrowCutYield = fruitTypeYield * windrowCutFactor
                end
                thFruitTypeData.windrowCutFactor = windrowCutFactor
            else
                thFruitTypeData.windrowCutYield = fruitTypeYield
                thFruitTypeData.windrowCutFactor = 1
            end
        end
        local maxNumStates = 0
        local numStateChannels = THUtils.toNumber(fruitTypeDesc.numStateChannels, true) or -1
        if numStateChannels > 0 then
            maxNumStates = (2 ^ numStateChannels) - 1
        end
        thFruitTypeData.maxNumStates = maxNumStates
        if maxNumStates > 0 then
            thFruitTypeData.maxGrowthState = THUtils.toNumber(fruitTypeDesc.numGrowthStates, true) or 0
            thFruitTypeData.minGrowthState = math.min(1, thFruitTypeData.maxGrowthState)
            thFruitTypeData.maxHarvestState = THUtils.toNumber(fruitTypeDesc.maxHarvestingGrowthState, true) or 0
            thFruitTypeData.minHarvestState = THUtils.toNumber(fruitTypeDesc.minHarvestingGrowthState, true) or thFruitTypeData.maxHarvestState
            thFruitTypeData.minForageState = THUtils.toNumber(fruitTypeDesc.minForageGrowthState, true) or 0
            if thFruitTypeData.minHarvestState > 1 and thFruitTypeData.minForageState > 0 then
                thFruitTypeData.maxForageState = math.max(thFruitTypeData.minForageState, thFruitTypeData.minHarvestState - 1)
            end
            thFruitTypeData.minPreparingState = THUtils.toNumber(fruitTypeDesc.minPreparingGrowthState, true) or 0
            thFruitTypeData.maxPreparingState = THUtils.toNumber(fruitTypeDesc.maxPreparingGrowthState, true) or thFruitTypeData.minPreparingState
            thFruitTypeData.minPreparedState = THUtils.toNumber(fruitTypeDesc.preparedGrowthState, true) or 0
            thFruitTypeData.maxPreparedState = thFruitTypeData.minPreparedState
            thFruitTypeData.minWitheredState = THUtils.toNumber(fruitTypeDesc.witheredState, true) or 0
            thFruitTypeData.maxWitheredState = thFruitTypeData.minWitheredState
        end
        local defaultCutState = THUtils.toNumber(fruitTypeDesc.cutState, true) or 0
        if defaultCutState > 0 then
            table.insert(thFruitTypeData.cutStates, defaultCutState)
        end
        if fruitTypeDesc.cutStates ~= nil then
            for fruitState in pairs(fruitTypeDesc.cutStates) do
                if fruitState > 0 and fruitState ~= defaultCutState then
                    table.insert(thFruitTypeData.cutStates, fruitState)
                end
            end
        end
        local numCutStates = #thFruitTypeData.cutStates
        thFruitTypeData.numCutStates = numCutStates
        if numCutStates > 0 then
            THUtils.sortTable(thFruitTypeData.cutStates, function(pa, pb)
                return pa < pb
            end)
            thFruitTypeData.minCutState = thFruitTypeData.cutStates[1]
            thFruitTypeData.maxCutState = thFruitTypeData.cutStates[numCutStates]
        end
        if maxNumStates > 0 then
            local finalState = thFruitTypeData.maxCutState
            if finalState <= 0 then
                finalState = thFruitTypeData.maxWitheredState
            end
            if finalState > 0 and finalState < maxNumStates then
                thFruitTypeData.minExtraState = math.min(maxNumStates, finalState + 1)
                thFruitTypeData.maxExtraState = maxNumStates
            end
        end
        local numHaulmStates = 0
        local numStateChannelsHaulm = THUtils.toNumber(fruitTypeDesc.numStateChannelsHaulm, true) or -1
        if numStateChannelsHaulm > 0 then
            numHaulmStates = (2 ^ numStateChannelsHaulm) - 1
        end
        thFruitTypeData.numHaulmStates = numHaulmStates
        self.fruitTypes[fruitTypeIndex] = thFruitTypeData
        if fruitTypeDesc.nameToGrowthState ~= nil then
            for fruitStateId in pairs(fruitTypeDesc.nameToGrowthState) do
                if type(fruitStateId) == THValueType.STRING then
                    fruitStateId = string.upper(fruitStateId)
                end
                self.validFruitStateNames[fruitStateId] = true
            end
        end
        self.maxFruitState = math.max(self.maxFruitState, thFruitTypeData.maxNumStates)
        self.maxFruitHaulmState = math.max(self.maxFruitHaulmState, thFruitTypeData.numHaulmStates)
    end
    return true
end
function THCore.initMapTypeData(self)
    local success = true
    local function getMapTypeData(pSelf, pId, pIdIsValue)
        if THUtils.argIsValid(not pIdIsValue or pIdIsValue == true, "idIsValue", pIdIsValue)
            and pId ~= nil
        then
            if pIdIsValue then
                return pSelf.byValue[pId]
            elseif type(pId) == THValueType.STRING then
                local mapTypeData = pSelf.byId[pId]
                if mapTypeData == nil then
                    mapTypeData = pSelf.byId[pId:upper()]
                end
                return mapTypeData
            else
                return pSelf.byIndex[pId]
            end
        end
    end
    local function addMapTypeData(pTypesName, pTypeId, pTypeIndex, pTypeValue)
        local mapTypeData = {
            id = pTypeId,
            index = pTypeIndex,
            value = pTypeValue
        }
        self[pTypesName].byId[pTypeId] = mapTypeData
        self[pTypesName].byIndex[pTypeIndex] = mapTypeData
        self[pTypesName].byValue[pTypeValue] = mapTypeData
    end
    local groundTypes = self.groundTypes
    local groundTypeList = FieldGroundType.getAll()
    groundTypes.byId = {}
    groundTypes.byIndex = {}
    groundTypes.byValue = {}
    groundTypes.getData = getMapTypeData
    if groundTypeList ~= nil then
        addMapTypeData("groundTypes", "NONE", 0, 0)
        for typeId, typeIndex in pairs(groundTypeList) do
            local typeValue = FieldGroundType.getValueByType(typeIndex)
            if typeValue ~= nil then
                addMapTypeData("groundTypes", typeId, typeIndex, typeValue)
            end
        end
    end
    local sprayTypes = self.sprayTypes
    local sprayTypeList = FieldSprayType.getAll()
    sprayTypes.byId = {}
    sprayTypes.byIndex = {}
    sprayTypes.byValue = {}
    sprayTypes.getData = getMapTypeData
    if sprayTypeList ~= nil then
        addMapTypeData("sprayTypes", "NONE", 0, 0)
        for typeId, typeIndex in pairs(sprayTypeList) do
            local typeValue = FieldSprayType.getValueByType(typeIndex)
            if typeValue ~= nil then
                addMapTypeData("sprayTypes", typeId, typeIndex, typeValue)
            end
        end
    end
    local chopperTypes = self.chopperTypes
    local chopperTypeList = FieldChopperType.getAll()
    chopperTypes.byId = {}
    chopperTypes.byIndex = {}
    chopperTypes.byValue = {}
    chopperTypes.getData = getMapTypeData
    if chopperTypeList ~= nil then
        addMapTypeData("chopperTypes", "NONE", 0, 0)
        for typeId, typeIndex in pairs(chopperTypeList) do
            local typeValue = FieldChopperType.getValueByType(typeIndex)
            if typeValue ~= nil then
                addMapTypeData("chopperTypes", typeId, typeIndex, typeValue)
            end
        end
    end
    return success
end
function THCore.load(self)
    return true
end
function THCore.loadMapData(self)
    local success = true
    self.hasMapConfig = false
    if self.xmlSchema ~= nil then
        local mapXMLFile, mapXMLFilename, customEnv, baseDirectory = self:loadMapXMLFile("THConfigXML_" .. self.dataKey, self.dataKey, self.xmlSchema, true)
        if mapXMLFile ~= nil then
            success = self:loadFromMapXML(mapXMLFile, self.xmlKey, customEnv, baseDirectory)
            if success then
                THUtils.displayMsg(THMessage.LOADING, mapXMLFilename)
                self.hasMapConfig = true
            else
                THUtils.errorMsg(nil, THMessage.LOADING_ERROR, mapXMLFilename)
            end
            THUtils.deleteXMLFile(mapXMLFile)
        end
    end
    return success
end
function THCore.loadFromMapXML(self, xmlFile, xmlKey, customEnv, baseDirectory)
    customEnv = customEnv or self.modName
    baseDirectory = baseDirectory or self.modPath
    if THUtils.argIsValid(type(xmlFile) == THValueType.TABLE, "xmlFile", xmlFile)
        and THUtils.argIsValid(type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(type(customEnv) == THValueType.STRING, "customEnv", customEnv)
        and THUtils.argIsValid(type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        return true
    end
    return false
end
function THCore.onSetMissionInfo(self, mission, missionInfo, missionDynamicInfo)
    self.mission = mission
    self.missionInfo = missionInfo
end
function THCore.getMapXMLFilename(self, xmlKey, useMapSettings, forceMapSettings)
    if THUtils.argIsValid(xmlKey == nil or (type(xmlKey) == THValueType.STRING and xmlKey ~= ""), "xmlKey", xmlKey)
        and THUtils.argIsValid(not useMapSettings or useMapSettings == true, "useMapSettings", useMapSettings)
        and THUtils.argIsValid(not forceMapSettings or forceMapSettings == true, "forceMapSettings", forceMapSettings)
        and self.missionInfo ~= nil and self.missionInfo.map ~= nil
    then
        local mapDesc = self.missionInfo.map
        local mapXMLFilename = nil
        local customEnv = nil
        local baseDirectory = nil
        local isMapSettings = false
        if useMapSettings
            and mapDesc.id ~= nil and mapDesc.id ~= ""
        then
            local thMapSettings = self.thMapSettings
            if thMapSettings ~= nil and fileExists(thMapSettings.xmlFilename) then
                local settingsXMLFile = THUtils.loadXMLFile("THMapSettingsXML", thMapSettings.xmlFilename, nil, true)
                local settingsXMLKey = thMapSettings.xmlRootKey .. ".maps." .. mapDesc.id
                if settingsXMLFile ~= nil then
                    if THUtils.hasXMLProperty(settingsXMLFile, settingsXMLKey) then
                        mapXMLFilename = THUtils.getXMLValue(settingsXMLFile, XMLValueType.STRING, settingsXMLKey, "#filename")
                        if mapXMLFilename ~= nil then
                            mapXMLFilename = THUtils.getFilename(mapXMLFilename, thMapSettings.modPath)
                            if THUtils.getFileExists(mapXMLFilename) then
                                customEnv = thMapSettings.modName
                                baseDirectory = thMapSettings.modPath
                                isMapSettings = true
                            else
                                mapXMLFilename = nil
                            end
                        end
                    end
                    THUtils.deleteXMLFile(settingsXMLFile)
                end
            end
        end
        if mapXMLFilename == nil and not forceMapSettings
            and mapDesc.isModMap
            and mapDesc.mapXMLFilename ~= nil
            and mapDesc.mapXMLFilename ~= ""
        then
            mapXMLFilename = THUtils.getFilename(mapDesc.mapXMLFilename, mapDesc.baseDirectory, true)
            if mapXMLFilename ~= nil then
                if THUtils.getFileExists(mapXMLFilename) then
                    customEnv = mapDesc.customEnvironment
                    baseDirectory = mapDesc.baseDirectory
                else
                    mapXMLFilename = nil
                end
            end
        end
        if mapXMLFilename ~= nil then
            local dataXMLFilename = nil
            if xmlKey == nil then
                dataXMLFilename = mapXMLFilename
            else
                local dataXMLKey = "map." .. xmlKey
                local mapXMLFile = THUtils.loadXMLFile("THMapXML", mapXMLFilename, nil, true)
                if mapXMLFile ~= nil then
                    dataXMLFilename = THUtils.getXMLValue(mapXMLFile, XMLValueType.STRING, dataXMLKey, "#filename")
                    if dataXMLFilename ~= nil then
                        dataXMLFilename = THUtils.getFilename(dataXMLFilename, baseDirectory)
                        if not THUtils.getFileExists(dataXMLFilename) then
                            dataXMLFilename = nil
                        end
                    end
                    THUtils.deleteXMLFile(mapXMLFile)
                end
            end
            if dataXMLFilename ~= nil then
                return dataXMLFilename, customEnv, baseDirectory, isMapSettings
            end
        end
    end
end
function THCore.loadMapXMLFile(self, xmlName, xmlKey, xmlSchema, useMapSettings, forceMapSettings)
    if THUtils.argIsValid(type(xmlName) == THValueType.STRING and xmlName ~= "", "xmlName", xmlName) then
        local dataXMLFilename, customEnv, baseDirectory, isMapSettings = self:getMapXMLFilename(xmlKey, useMapSettings, forceMapSettings)
        if dataXMLFilename ~= nil
            and THUtils.argIsValid(type(xmlSchema) == THValueType.TABLE or not xmlSchema or xmlSchema == true, "xmlSchema", xmlSchema)
        then
            local dataXMLFile = THUtils.loadXMLFile(xmlName, dataXMLFilename, nil, xmlSchema)
            if dataXMLFile ~= nil then
                return dataXMLFile, dataXMLFilename, customEnv, baseDirectory, isMapSettings
            end
        end
    end
end
function THCore.getFruitType(self, fruitType, verbose)
    verbose = THUtils.validateArg(not verbose or verbose == true, "verbose", verbose, true)
    local fruitTypeManager = self.fruitTypeManager
    local fruitTypeIdType = type(fruitType)
    local fruitTypeDesc = nil
    if fruitTypeIdType == THValueType.STRING then
        fruitTypeDesc = fruitTypeManager:getFruitTypeByName(fruitType)
    elseif fruitTypeIdType == THValueType.NUMBER then
        fruitTypeDesc = fruitTypeManager:getFruitTypeByIndex(fruitType)
    elseif fruitTypeIdType == THValueType.TABLE and fruitType.name ~= nil then
        fruitTypeDesc = fruitTypeManager:getFruitTypeByName(fruitType.name)
    elseif fruitType ~= nil then
        verbose = true
    end
    if fruitTypeDesc ~= nil then
        local thFruitTypeData = self.fruitTypes[fruitTypeDesc.index]
        if thFruitTypeData ~= nil then
            return thFruitTypeData
        end
    end
    if verbose then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "fruitType", fruitType)
    end
end
function THCore.getFruitTypes(self)
    return self.fruitTypes, #self.fruitTypes
end
function THCore.getFruitGrowthStateParams(self, fruitStateId, allowFactor)
    allowFactor = THUtils.validateArg(not allowFactor or allowFactor == true, "allowFactor", allowFactor, false)
    local stateIdType = type(fruitStateId)
    if stateIdType == THValueType.NUMBER then
        local stateValue = THUtils.floor(fruitStateId)
        if allowFactor then
            stateValue = THUtils.clamp(stateValue, -self.maxFruitState, self.maxFruitState)
        else
            stateValue = THUtils.clamp(stateValue, 0, self.maxFruitState)
        end
        if stateValue < 0 then
            return nil, nil, nil, true, fruitStateId
        end
        return stateValue
    elseif stateIdType == THValueType.STRING then
        local absFruitStateId = fruitStateId
        fruitStateId = fruitStateId:upper()
        if self.validFruitStateNames[fruitStateId] then
            return fruitStateId
        end
        if allowFactor and string.find(fruitStateId, "^[+-]") then
            local factorValues = THUtils.splitString(fruitStateId, "|")
            if factorValues ~= nil then
                local stateFactor = THUtils.toNumber(factorValues[1], true)
                if stateFactor ~= nil then
                    stateFactor = THUtils.clamp(stateFactor, -self.maxFruitState, self.maxFruitState)
                    local stateLimit = factorValues[2]
                    if stateLimit == nil then
                        return nil, nil, nil, true, stateFactor
                    end
                    local otherStateName = self:getFruitGrowthStateParams(stateLimit)
                    if otherStateName ~= nil then
                        return stateLimit, nil, nil, true, stateFactor
                    end
                end
            end
        else
            local fruitState = THUtils.toNumber(fruitStateId, true)
            if fruitState ~= nil then
                fruitState = THUtils.clamp(fruitState, 0, self.maxFruitState)
                return fruitState
            end
            local _, _, absStateName, offset2 = string.find(fruitStateId, "(.+)([+-]%d+)")
            if absStateName == nil or offset2 == nil then
                absStateName = fruitStateId
                offset2 = 0
            else
                offset2 = THUtils.toNumber(offset2, true)
                if offset2 == nil then
                    THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "fruitStateId", fruitStateId)
                    return
                end
                offset2 = THUtils.clamp(offset2, -self.maxFruitState, self.maxFruitState)
            end
            local _, _, stateName, offset1 = string.find(absStateName, "(.+)_(%d+)")
            if stateName == nil or offset1 == nil
                or (stateName ~= nil and THFruitState:getValue(stateName) == nil)
            then
                _, _, stateName, offset1 = string.find(absStateName, "(.+)_(%a+)")
                if stateName == nil or offset1 == nil
                    or THFruitState:getValue(stateName) == nil
                then
                    if THFruitState:getValue(absStateName) == nil then
                        return absFruitStateId
                    end
                    stateName = absStateName
                    offset1 = 1
                end
            end
            if offset1 == "MIN" then
                offset1 = 1
            elseif offset1 == "MAX" then
                offset1 = self.maxFruitState
            else
                offset1 = THUtils.toNumber(offset1, true)
                if offset1 == nil then
                    THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "fruitStateId", fruitStateId)
                    return
                end
                offset1 = THUtils.clamp(offset1, 1, self.maxFruitState)
            end
            if THFruitState:getValue(stateName) ~= nil then
                return stateName, offset1, offset2
            end
        end
    end
end
function THCore.getFruitGrowthState(self, fruitType, fruitStateId, allowFactor, verbose)
    local thFruitTypeData = self:getFruitType(fruitType, verbose)
    if thFruitTypeData ~= nil then
        local stateName, offset1, offset2, isFactor, stateFactor = self:getFruitGrowthStateParams(fruitStateId, allowFactor)
        local maxNumStates = thFruitTypeData.maxNumStates or 0
        if isFactor == true then
            if stateFactor == nil then
                return
            end
            fruitStateId = stateName
            stateFactor = THUtils.clamp(stateFactor, -maxNumStates, maxNumStates)
            if fruitStateId ~= nil then
                stateName, offset1, offset2 = self:getFruitGrowthStateParams(fruitStateId)
            else
                stateName, offset1, offset2 = nil, nil, nil
            end
        end
        if stateName == nil then
            return thFruitTypeData, nil, isFactor == true, stateFactor
        end
        local fruitStateType = type(stateName)
        local fruitState = nil
        if fruitStateType == THValueType.NUMBER then
            fruitState = THUtils.clamp(stateName, 0, thFruitTypeData.maxNumStates)
        elseif fruitStateType == THValueType.STRING then
            fruitState = thFruitTypeData.desc:getGrowthStateByName(stateName)
            if fruitState == nil then
                local stateIndex = THFruitState:getValue(stateName)
                if stateIndex ~= nil then
                    local minState = 0
                    local maxState = maxNumStates
                    if stateIndex == THFruitState.ZERO then
                        minState = 0
                        maxState = 0
                    elseif stateIndex == THFruitState.MIN then
                        minState = 1
                        maxState = 1
                    elseif stateIndex == THFruitState.GROWTH then
                        minState = thFruitTypeData.minGrowthState
                        maxState = thFruitTypeData.maxGrowthState
                    elseif stateIndex == THFruitState.FORAGE then
                        minState = thFruitTypeData.minForageState
                        maxState = thFruitTypeData.maxForageState
                    elseif stateIndex == THFruitState.HARVEST then
                        minState = thFruitTypeData.minHarvestState
                        maxState = thFruitTypeData.maxHarvestState
                    elseif stateIndex == THFruitState.WITHERED then
                        minState = thFruitTypeData.minWitheredState
                        maxState = thFruitTypeData.maxWitheredState
                    elseif stateIndex == THFruitState.PREPARING then
                        minState = thFruitTypeData.minPreparingState
                        maxState = thFruitTypeData.maxPreparingState
                    elseif stateIndex == THFruitState.PREPARED then
                        minState = thFruitTypeData.minPreparedState
                        maxState = thFruitTypeData.maxPreparedState
                    elseif stateIndex == THFruitState.CUT then
                        minState = thFruitTypeData.minCutState
                        maxState = thFruitTypeData.maxCutState
                    elseif stateIndex == THFruitState.EXTRA then
                        minState = thFruitTypeData.minExtraState
                        maxState = thFruitTypeData.maxExtraState
                    elseif stateIndex == THFruitState.MAX then
                        minState = maxNumStates
                        maxState = maxNumStates
                    else
                        THUtils.errorMsg(true, "Could not find information for growth state: %s", fruitState)
                        return
                    end
                    if minState >= 0 and maxState >= 0 then
                        fruitState = -1
                        if offset1 == nil or offset1 <= 1 then
                            fruitState = minState
                        else
                            if stateIndex == THFruitState.CUT then
                                fruitState = thFruitTypeData.cutStates[offset1] or maxState
                            else
                                fruitState = minState + (offset1 - 1)
                            end
                            fruitState = THUtils.clamp(fruitState, minState, maxState)
                        end
                        if fruitState ~= nil and fruitState >= 0 then
                            fruitState = THUtils.clamp(fruitState + (offset2 or 0), 0, maxNumStates)
                        end
                    end
                end
            end
        end
        if fruitState ~= nil and fruitState >= 0 then
            return thFruitTypeData, fruitState, isFactor == true, stateFactor
        end
        if verbose then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "fruitStateId", fruitStateId)
        end
    end
end
function THCore.getFruitHaulmState(self, fruitType, haulmStateId)
    local haulmStateIdType = type(haulmStateId)
    local haulmState = nil
    if haulmStateIdType == THValueType.STRING then
        haulmStateId = string.upper(haulmStateId)
        if haulmStateId == "MIN" or haulmStateId == "NONE" then
            haulmState = 0
        elseif haulmStateId == "MAX" then
            haulmState = self.maxFruitHaulmState
        else
            haulmState = THUtils.toNumber(haulmStateId, true)
        end
    elseif haulmStateIdType == THValueType.NUMBER
        and math.floor(haulmStateId) == haulmStateId
    then
        haulmState = haulmStateId
    end
    if haulmState ~= nil and haulmState >= 0 then
        if fruitType == nil then
            haulmState = THUtils.clamp(haulmState, 0, self.maxFruitHaulmState)
            return nil, haulmState
        end
        local thFruitTypeData = self:getFruitType(fruitType)
        if thFruitTypeData == nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "fruitType", fruitType)
            return
        end
        haulmState = THUtils.clamp(haulmState, 0, thFruitTypeData.numHaulmStates)
        return thFruitTypeData, haulmState
    end
end
function THCore.getFillType(self, fillType, verbose)
    verbose = THUtils.validateArg(not verbose or verbose == true, "verbose", verbose, true)
    local fillTypeManager = self.fillTypeManager
    local fillTypeIdType = type(fillType)
    local fillTypeDesc = nil
    if fillTypeIdType == THValueType.STRING then
        fillTypeDesc = fillTypeManager:getFillTypeByName(fillType)
    elseif fillTypeIdType == THValueType.NUMBER then
        fillTypeDesc = fillTypeManager:getFillTypeByIndex(fillType)
    elseif fillTypeIdType == THValueType.TABLE and fillType.name ~= nil then
        fillTypeDesc = fillTypeManager:getFillTypeByName(fillType)
    elseif fillType ~= nil then
        verbose = true
    end
    if fillTypeDesc ~= nil then
        local thFillTypeData = self.fillTypes[fillTypeDesc.index]
        if thFillTypeData ~= nil then
            return thFillTypeData
        end
    end
    if verbose then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "fillType", fillType)
    end
end
function THCore.getFillTypes(self)
    return self.fillTypes, #self.fillTypes
end