local modName, modDir = g_currentModName, g_currentModDirectory
local isDebug = modDir:contains(".+%_dev/$")

FoliageSystem.loadMapData = Utils.appendedFunction(FoliageSystem.loadMapData, function(self, xmlFile, missionInfo, baseDirectory)
    local settingsDirectory = g_modSettingsDirectory .. modName .. "/"
    if not folderExists(settingsDirectory) then
        createFolder(settingsDirectory)
    end
    settingsDirectory = settingsDirectory .. missionInfo.mapId .. "/"

    local mapFilename = getXMLString(xmlFile, "map.filename")
    mapFilename = Utils.getFilename(mapFilename, baseDirectory)
    local mapFile = XMLFile.loadIfExists("i3dMap", mapFilename)
    if mapFile == nil then
        Logging.error("[%s] Failed to load map file (%s). Skipping shop implementation!", modName, mapFilename)
        return
    end
    local mapMD5 = getMD5(saveXMLFileToMemory(mapFile.handle))
    local modVersion = g_modManager:getModByName(modName).version

    if folderExists(settingsDirectory) then
        local paintableFoliagesFile = XMLFile.loadIfExists("paintableFoliages", settingsDirectory .. "paintableFoliages.xml")
        if paintableFoliagesFile ~= nil then
            local version = paintableFoliagesFile:getString("paintableFoliages#modVersion")
            local hash = paintableFoliagesFile:getString("paintableFoliages#mapHash")
            if mapMD5 ~= hash or modVersion ~= version or isDebug then
                deleteFolder(settingsDirectory)
            end
            paintableFoliagesFile:delete()
        end
    end

    if not folderExists(settingsDirectory) then
        Logging.info("[%s] Creating foliage brushes. Please stand by.", modName)

        createFolder(settingsDirectory)
        if not folderExists(settingsDirectory) then
            Logging.error("[%s] Failed to create mod settings directory (%s). Skipping shop implementation!", modName, settingsDirectory)
            return
        end

        local templateFile = XMLFile.create("foliageStoreTemplate", "", "placeable", Placeable.xmlSchema)
        templateFile:setValue("placeable#type", "bush")
        templateFile:setValue("placeable.storeData.species", "PLACEABLE")
        templateFile:setValue("placeable.storeData.brand", "NONE")
        templateFile:setValue("placeable.storeData.category", "placeableMisc")
        templateFile:setValue("placeable.storeData.price", 20)
        templateFile:setValue("placeable.storeData.functions.function(0)", "$l10n_function_foliage")
        templateFile:setValue("placeable.storeData.brush.type", "foliage")
        templateFile:setValue("placeable.storeData.brush.category", "landscaping")
        templateFile:setValue("placeable.storeData.brush.tab", "plants")

        local filesMap = {}
        mapFile:iterate("i3D.Files.File", function(_, key)
            local filename = mapFile:getString(key .. "#filename")
            if filename:contains("%.xml$") then
                filesMap[mapFile:getInt(key .. "#fileId")] = filename
            end
        end)

        local paintableFoliagesFile = XMLFile.create("paintableFoliages", settingsDirectory .. "paintableFoliages.xml", "paintableFoliages")
        local paintableFoliageIdx = 0
        paintableFoliagesFile:setString("paintableFoliages#modVersion", modVersion)
        paintableFoliagesFile:setString("paintableFoliages#mapHash", mapMD5)

        local i3dContainer = nil
        local cameraBaseNode = nil
        local camera = nil
        local light = nil

        if g_client ~= nil then
            i3dContainer = createTransformGroup("foliageI3dContainer")
            link(getRootNode(), i3dContainer)
            setTranslation(i3dContainer, -100, -100, -100)

            cameraBaseNode = createTransformGroup("StoreIconRenderCameraBaseNode")
            camera = createCamera("StoreIconRenderCamera", math.rad(60), 0.1, 1000)
            link(cameraBaseNode, camera)
            link(i3dContainer, cameraBaseNode)
            setRotation(cameraBaseNode, math.rad(-15), math.rad(45), 0)
            g_cameraManager:addCamera(camera, nil, true)
            g_cameraManager:setActiveCamera(camera)

            light = createLightSource("StoreIconRenderLight", LightType.DIRECTIONAL, 0.75, 0.75, 0.75, 100)
            setLightShadowMap(light, true, 512)
            link(i3dContainer, light)
            setRotation(light, math.rad(-95), math.rad(70), math.rad(-5))
        end

        mapFile:iterate("i3D.Scene.TerrainTransformGroup.Layers.FoliageSystem.FoliageMultiLayer", function(_, key)
            mapFile:iterate(key .. ".FoliageType", function(_, key)
                local foliageLayer = mapFile:getString(key .. "#name")
                local foliageFilename = filesMap[mapFile:getInt(key .. "#foliageXmlId")]
                if foliageLayer ~= nil and foliageFilename ~= nil then
                    foliageFilename = Utils.getFilename(foliageFilename, Utils.getDirectory(mapFilename))
                    local foliageFile = XMLFile.loadIfExists("foliage", foliageFilename, FruitTypeDesc.xmlSchema)
                    if foliageFile == nil then
                        return
                    end

                    if foliageFile:hasProperty("foliageType.fruitType")
                        or foliageFile:getValue("foliageType.foliageLayer(0).foliageState(0)#name") == "invisible" then
                            return
                    end

                    local storeI3dFile = foliageFile:getValue("foliageType.foliageLayer(0)#shapeSource", "")
                    storeI3dFile = Utils.getFilename(storeI3dFile, Utils.getDirectory(foliageFilename))

                    local i3dNode = nil
                    if g_client ~= nil and fileExists(storeI3dFile) then
                        i3dNode = loadI3DFile(storeI3dFile, false, false, false)
                    end

                    foliageFile:iterate("foliageType.foliageLayer(0).foliageState", function(idx, key)
                        for _, storeItem in pairs(g_storeManager.items) do
                            if storeItem.brush
                                and storeItem.brush.type == "foliage"
                                and storeItem.brush.parameters[1] == foliageLayer
                                and storeItem.brush.parameters[2] == tostring(idx) then
                                    return
                            end
                        end

                        local foliageName = foliageFile:getValue(key .. "#name")
                        local storeName = foliageLayer .. " - " .. foliageName
                        storeName = storeName:gsub("(%w)(%u)", "%1 %2"):gsub("%f[%a].", string.upper)

                        local storeIcon = "store_" .. storeName:gsub("[%s-_]+", "") .. ".dxt" -- dxt won't throw warnings without compression
                        local renderNodePath = foliageFile:getString(key .. ".foliageShape(0).foliageLod(0)#blockShape", "")
                        renderNodePath = renderNodePath:gsub(">", "|")

                        if i3dNode ~= nil then
                            local objectNode = clone(I3DUtil.indexToObject(i3dNode, renderNodePath) or i3dNode, false, false, false)
                            link(i3dContainer, objectNode)
                            local posX, posY, posZ, radius = getShapeBoundingSphere(objectNode)
                            radius = radius * math.max(getScale(objectNode))
                            setTranslation(cameraBaseNode, posX, posY + radius * 0.1, posZ)
                            setTranslation(camera, 0, 0, radius * 2)

                            if not renderScreenshot(settingsDirectory .. storeIcon, 512, 512, 1, "raw_alpha", 2, 0, 0, 0, 0, 0, 15, false) then
                                storeIcon = "$data/shared/warningStripes_diffuse.dds"
                            end
                            delete(objectNode)
                        else
                            storeIcon = "$data/shared/warningStripes_diffuse.dds"
                        end
                        templateFile:setValue("placeable.storeData.name", storeName)
                        templateFile:setValue("placeable.storeData.image", storeIcon)
                        templateFile:setValue("placeable.storeData.brush.parameters.parameter(0)", foliageLayer)
                        templateFile:setValue("placeable.storeData.brush.parameters.parameter(1)", tostring(idx))
                        templateFile:setValue("placeable.base.filename", storeI3dFile:gsub("^data/", "$data/"))

                        templateFile:saveTo(settingsDirectory .. "paintable_" .. storeName:gsub("[%s-_]+", "") .. ".xml")
                    end)
                    if i3dNode ~= nil then delete(i3dNode) end

                    local paintableFoliagesKey = string.format("paintableFoliages.paintableFoliage(%d)", paintableFoliageIdx)
                    paintableFoliagesFile:setString(paintableFoliagesKey .. "#layerName", foliageLayer)
                    paintableFoliagesFile:setInt(paintableFoliagesKey .. "#startStateChannel", foliageFile:getValue("foliageType.foliageLayer(0)#densityMapChannelOffset", 0))
                    paintableFoliagesFile:setInt(paintableFoliagesKey .. "#numStateChannels", foliageFile:getValue("foliageType.foliageLayer(0)#numDensityMapChannels", 0))
                    paintableFoliageIdx = paintableFoliageIdx + 1

                    foliageFile:delete()
                end
            end)
        end)

        if light ~= nil then delete(light) end
        if camera ~= nil then
            g_cameraManager:setActiveCamera(g_cameraManager.defaultCameraNode)
            g_cameraManager:removeCamera(camera)
            delete(camera)
        end
        if cameraBaseNode ~= nil then delete(cameraBaseNode) end
        if i3dContainer ~= nil then delete(i3dContainer) end

        paintableFoliagesFile:save()
        paintableFoliagesFile:delete()
        templateFile:delete()
    end
    mapFile:delete()

    local paintableFoliagesFile = XMLFile.loadIfExists("paintableFoliages", settingsDirectory .. "paintableFoliages.xml")
    if paintableFoliagesFile == nil then
        Logging.error("[%s] Failed to load paintables file (%s). Skipping shop implementation!", modName, settingsDirectory .. "paintableFoliages.xml")
        return
    end
    paintableFoliagesFile:iterate("paintableFoliages.paintableFoliage", function(idx, key)
        local layerName = paintableFoliagesFile:getString(key .. "#layerName")
        if self:getFoliagePaintByName(layerName) == nil then
            self.paintableFoliages[#self.paintableFoliages + 1] = {
                id = #self.paintableFoliages + 1,
                layerName = layerName,
                startStateChannel = paintableFoliagesFile:getInt(key .. "#startStateChannel"),
                numStateChannels = paintableFoliagesFile:getInt(key .. "#numStateChannels"),
                state = 0
            }
        end
    end)

    local paintables = Files.getFilesRecursive(settingsDirectory)
    for _, paintable in pairs(paintables) do
        if not paintable.isDirectory then
            if paintable.filename:contains("^paintable_.+%.xml$") then
                g_storeManager:loadItem(paintable.filename, settingsDirectory, modName, true)
            end
        end
    end
end)